<?php

namespace App\Console\Commands;

use App\Models\Job;
use Illuminate\Console\Command;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Log;

class CloseExpiredJobs extends Command
{
    protected $signature = 'jobs:close-expired {--test : Run in test mode to simulate past deadlines}';
    protected $description = 'Close jobs that are past their deadline';

    public function handle()
    {
        $now = Carbon::now('UTC');
        $this->info("Current UTC time: $now");

        // Check if test mode is enabled
        if ($this->option('test')) {
            $this->info('Running in test mode. Simulating past deadlines...');
            $testDate = Carbon::now('UTC')->subDays(30); // Simulate 30 days ago
            $expiredJobs = Job::whereNotNull('deadline')
                ->where('deadline', '<', $testDate)
                ->get();
        } else {
            $expiredJobs = Job::whereNotNull('deadline')
                ->where('deadline', '<', $now)
                ->get();
        }

        if ($expiredJobs->isEmpty()) {
            $this->info('No expired jobs found.');
            Log::info('No expired jobs found at ' . $now);
        } else {
            foreach ($expiredJobs as $job) {
                $jobDeadline = Carbon::parse($job->deadline)->setTimezone('UTC');
                Log::info("Processing job ID {$job->Job_id}: Deadline {$jobDeadline} vs Now {$now}");
                if ($job->update(['Status' => 'closed'])) {
                    Log::info("Job ID {$job->Job_id} closed successfully.");
                    $this->info("Job ID {$job->Job_id} closed.");
                } else {
                    Log::warning("Failed to update job ID {$job->Job_id}.");
                    $this->error("Failed to update job ID {$job->Job_id}.");
                }
            }
        }
    }
}
